<?php
/**
 * Functions
 *
 * @package     AutomatorWP\BuddyBoss\Functions
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Register custom component
 *
 * @since 1.0.0
 *
 * @param array $component_names
 *
 * @return array
 */
function automatorwp_buddyboss_register_notifications_component( $component_names = array() ) {

    // Force $component_names to be an array
    if ( ! is_array( $component_names ) ) {
        $component_names = array();
    }

    // Add 'automatorwp' component
    array_push( $component_names, 'automatorwp' );

    return $component_names;

}
add_filter( 'bp_notifications_get_registered_components', 'automatorwp_buddyboss_register_notifications_component' );

/**
 * Options callback for select2 fields assigned to groups
 *
 * @since 1.0.0
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_buddyboss_options_cb_group( $field ) {

    // Setup vars
    $value = $field->escaped_value;
    $none_value = 'any';
    $none_label = __( 'any group', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );

    if( ! empty( $value ) ) {
        if( ! is_array( $value ) ) {
            $value = array( $value );
        }

        foreach( $value as $group_id ) {

            // Skip option none
            if( $group_id === $none_value ) {
                continue;
            }

            $options[$group_id] = automatorwp_buddyboss_get_group_title( $group_id );
        }
    }

    return $options;

}

/**
 * Get the group title
 *
 * @since 1.0.0
 *
 * @param int $group_id
 *
 * @return string|null
 */
function automatorwp_buddyboss_get_group_title( $group_id ) {

    // Empty title if no ID provided
    if( absint( $group_id ) === 0 ) {
        return '';
    }

    // Bail if BuddyBoss function does not exist
    if ( ! function_exists( 'groups_get_group' ) ) {
        return '';
    }

    $group = groups_get_group( $group_id );

    return $group->name;

}

/**
 * Options callback for select fields assigned to group roles
 *
 * @since 1.0.0
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_buddyboss_group_roles_options_cb( $field ) {


    $none_value = 'any';
    $none_label = __( 'any group role', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );

    // Bail if BuddyBoss function does not exist
    if ( ! function_exists( 'bp_groups_get_group_roles' ) ) {
        return $options;
    }

    $roles = bp_groups_get_group_roles( );

    foreach( $roles as $rol => $rol_obj ) {
        $options[$rol] = $rol_obj->name;
    }

    return $options;

}

/**
 * Options callback for select fields assigned to member types
 *
 * @since 1.0.0
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_buddyboss_member_types_options_cb( $field ) {

    $none_value = 'any';
    $none_label = __( 'any profile type', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );

    $member_types = bp_get_member_types( array(), 'objects' );

    foreach( $member_types as $member_type => $member_type_obj ) {
        $options[$member_type] = $member_type_obj->labels['singular_name'];
    }

    return $options;

}

/**
 * Options callback for select2 fields assigned to profile fields
 *
 * @since 1.0.0
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_buddyboss_options_cb_profile_field( $field ) {

    // Setup vars
    $value = $field->escaped_value;
    $none_value = 'any';
    $none_label = __( 'any field', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );

    if( ! empty( $value ) ) {
        if( ! is_array( $value ) ) {
            $value = array( $value );
        }

        foreach( $value as $profile_field_id ) {

            // Skip option none
            if( $profile_field_id === $none_value ) {
                continue;
            }

            $options[$profile_field_id] = automatorwp_buddyboss_get_profile_field_title( $profile_field_id );
        }
    }

    return $options;

}

/**
 * Get the profile field title
 *
 * @since 1.0.0
 *
 * @param int $profile_field_id
 *
 * @return string
 */
function automatorwp_buddyboss_get_profile_field_title( $profile_field_id ) {

    global $wpdb;

    // Empty title if no ID provided
    if( absint( $profile_field_id ) === 0 ) {
        return '';
    }

    $prefix = automatorwp_buddyboss_get_table_prefix();

    $profile_field_name = $wpdb->get_var( "SELECT name FROM {$prefix}bp_xprofile_fields WHERE id = {$profile_field_id}");

    return ( $profile_field_name ? $profile_field_name : '' );

}

/**
 * Helper function to get the table prefix
 *
 * @since 1.0.0
 *
 * @return string
 */
function automatorwp_buddyboss_get_table_prefix() {

    global $wpdb;

    if( function_exists( 'bp_core_get_table_prefix' ) ) {
        return bp_core_get_table_prefix();
    }

    return $wpdb->prefix;

}

/**
 * Options callback for select2 fields assigned to group types
 *
 * @since 1.3.2
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_buddyboss_options_cb_group_type( $field ) {

    // Setup vars
    $value = $field->escaped_value;
    $none_value = 'any';
    $none_label = __( 'any type', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );

    if( ! empty( $value ) ) {
        if( ! is_array( $value ) ) {
            $value = array( $value );
        }

        foreach( $value as $group_type ) {
            // Skip option none
            if( $group_type === $none_value ) {
                continue;
            }

            $options[$group_type] = $group_type;
        }
    }

    return $options;

}

/**
 * Helper function to get the preview from a URL
 *
 * @since 1.5.0
 *
 * @param string $link
 *
 * @return array|false
 */
function automatorwp_buddyboss_get_link_preview( $link ) {

    // Bail if not is a valid URL
    if ( ! filter_var( $link, FILTER_VALIDATE_URL ) ) {
        return false;
    }

    $title = '';
    $description = '';
    $image_url = '';

    // Check if BP parse URL can take the information
    if( function_exists( 'bp_core_parse_url' ) ) {

        $bp_data = bp_core_parse_url( $link );

        $title = $bp_data['title'];
        $description = $bp_data['description'];
        $image_url = ( is_array( $bp_data['images'] ) && count( $bp_data['images'] ) ) ? $bp_data['images'][0] : '';

        // All information done so return it here without continue
        if( ! empty( $title ) && ! empty( $description ) && ! empty( $image_url ) ) {
            return array(
                'url' => $link,
                'title' => $title,
                'description' => $description,
                'image_url' => $image_url
            );
        }
    }

    // Extract HTML using curl
    $ch = curl_init();

    curl_setopt( $ch, CURLOPT_HEADER, 0 );
    curl_setopt( $ch, CURLOPT_RETURNTRANSFER, 1 );
    curl_setopt( $ch, CURLOPT_URL, $link );
    curl_setopt( $ch, CURLOPT_FOLLOWLOCATION, 1 );

    $data = curl_exec( $ch );
    curl_close( $ch );

    if( empty( $data ) ) {
        return false;
    }

    // Load HTML to DOM Object
    $dom = new DOMDocument();
    @$dom->loadHTML( $data );

    // Parse DOM to get Title
    if( empty( $title ) ) {
        $nodes = $dom->getElementsByTagName('title');

        if( $nodes->length > 0 ) {
            $title = $nodes->item(0)->nodeValue;
        }
    }

    // Parse DOM to get Meta Description
    $metas = $dom->getElementsByTagName('meta');

    for ($i = 0; $i < $metas->length; $i ++) {
        $meta = $metas->item( $i );

        // Description meta
        if ( $meta->getAttribute('name') == 'description' && empty( $description ) ) {
            $description = $meta->getAttribute('content');
        }

        // OG Metas (they have priority)
        if ( $meta->getAttribute('property') == 'og:title' ) {
            $title = $meta->getAttribute('content');
        }

        if ( $meta->getAttribute('property') == 'og:description' ) {
            $description = $meta->getAttribute('content');
        }

        if ( $meta->getAttribute('property') == 'og:image' ) {
            $image_url = $meta->getAttribute('content');
        }
    }

    // Parse DOM to get Images
    if( empty( $image_url ) ) {
        $images = $dom->getElementsByTagName('img');

        for ( $i = 0; $i < $images->length; $i ++) {
            $image = $images->item( $i );
            $src = $image->getAttribute( 'src' );

            if( filter_var( $src, FILTER_VALIDATE_URL ) ) {
                $image_url = $src;
                break;
            }
        }
    }

    return array(
        'url' => $link,
        'title' => $title,
        'description' => $description,
        'image_url' => $image_url
    );

}

/**
 * Get the media URLs assigned to an activity
 *
 * @since 1.5.0
 *
 * @param int $activity_id
 *
 * @return array
 */
function automatorwp_buddyboss_get_activity_media_urls( $activity_id ) {

    global $wpdb;
    $bp = buddypress();
    $urls = array();
    $activity_id = absint( $activity_id );

    $ids = bp_activity_get_meta( $activity_id, 'bp_media_ids', true );

    if ( ! isset( $ids ) || null === $ids || empty( $ids ) ) {
        $ids = bp_activity_get_meta( $activity_id, 'bp_media_ids', true );
    }

    if ( ! empty( $ids ) ) {

        $ids = explode( ',', $ids );

        foreach ( $ids as $id ){
    
            $attachment_id = $wpdb->get_var( "SELECT `attachment_id` from {$bp->media->table_name} WHERE id = {$id}" );
            $urls[] = bp_media_get_preview_image_url( $id, $attachment_id);
        }

    }
    return $urls;

}

/**
 * Get the video URLs assigned to an activity
 *
 * @since 1.5.0
 *
 * @param int $activity_id
 *
 * @return array
 */
function automatorwp_buddyboss_get_activity_video_urls( $activity_id ) {

    global $wpdb;
    $bp = buddypress();
    $urls = array();
    $activity_id = absint( $activity_id );

    $ids = bp_activity_get_meta( $activity_id, 'bp_video_ids', true );
    
    if ( ! empty( $ids ) ) {

        $ids = explode( ',', $ids );

        foreach ( $ids as $video_id ){
    
            $attachment_id = $wpdb->get_var( "SELECT `attachment_id` from {$bp->media->table_name} WHERE id = {$video_id}" );

            // Get the video link generated by BuddyBoss
            $video    = new BP_Video( $video_id );
            $media = bb_video_get_symlink( $video, $generate = true );

            $urls[] = $media;
        }

    }

    return $urls;

}

/**
 * Get the document URLs assigned to an activity
 *
 * @since 1.5.0
 *
 * @param int $activity_id
 *
 * @return array
 */
function automatorwp_buddyboss_get_activity_document_urls( $activity_id ) {

    global $wpdb;
    $bp = buddypress();
    $urls = array();
    $activity_id = absint( $activity_id );

    $ids = bp_activity_get_meta( $activity_id, 'bp_document_ids', true );

    if ( ! empty( $ids ) ) {

        $ids = explode( ',', $ids );

        foreach ( $ids as $document_id ){
    
            $attachment_id = $wpdb->get_var( "SELECT `attachment_id` from {$wpdb->prefix}bp_document WHERE id = {$document_id}" );

            $document_url = bp_document_download_link( $attachment_id, $document_id );
            $urls[] = $document_url;

        }

    }

    return $urls;

}

/**
 * Helper function to format notifications
 *
 * @since 1.5.9
 *
 * @param array $options
 * @param int $item_id
 * @param string $format
 * @param string $format
 * @param string $component_action
 *
 * @return string|array|false
 */
function automatorwp_buddyboss_format_notification( $options, $item_id, $format, $component_action ) {

    // Check if content is not empty
    if( ! empty( $options['content'] ) ) {

        if( $format === 'string' ) {
            // Text format

            if( $options['clickable'] ) {
                // Clickable notification

                // Set up the notification link
                $href = wp_nonce_url(
                    add_query_arg(
                        array(
                            'action' => 'automatorwp_buddyboss_mark_read',
                            'item_id' => $item_id,
                            'component_action' => $component_action
                        ),
                        $options['link']
                    ),
                    'automatorwp_buddyboss_mark_item_' . $item_id
                );

                return '<a href="' . $href . '">' . $options['content'] . '</a>';

            } else {
                // Text notification
                return $options['content'];
            }

        } else if( $format === 'object' ) {
            // Object format
            return array(
                'text' => $options['content'],
                'link' => $options['link']
            );
        }

    }

    return false;

}

/**
 * Mark as read notifications made by AutomatorWP
 *
 * @since 1.5.9
 */
function automatorwp_buddyboss_notifications_mark_read_handler() {

    if( ! isset( $_GET['action'] ) ) {
        return;
    }

    $action = sanitize_key( $_GET['action'] );

    // Bail if action is not for this function
    if ( $action !== 'automatorwp_buddyboss_mark_read' ) {
        return;
    }

    // Bail if no item ID is passed
    if ( empty( $_GET['item_id'] ) ) {
        return;
    }

    // Get required data
    $user_id  = get_current_user_id();
    $item_id = intval( $_GET['item_id'] );

    // Bail if nonce not passed
    if( ! isset( $_GET['_wpnonce'] ) ) {
        return;
    }

    // Check nonce
    if ( ! wp_verify_nonce( $_GET['_wpnonce'], 'automatorwp_buddyboss_mark_item_' . $item_id ) ) {
        return;
    } elseif ( ! current_user_can( 'edit_user', $user_id ) ) {
        return;
    }

    if( isset( $_GET['component_action'] ) && ! empty( $_GET['component_action'] ) ) {
        $success = bp_notifications_mark_notifications_by_item_id( $user_id, $item_id, 'automatorwp', $_GET['component_action'] );
    }

    $url = remove_query_arg( 'action' );
    $url = remove_query_arg( 'item_id', $url );
    $url = remove_query_arg( 'component_action', $url );
    $url = remove_query_arg( '_wpnonce', $url );

    // Redirect
    wp_safe_redirect( $url );

    // For good measure
    exit();

}
add_action( 'bp_init', 'automatorwp_buddyboss_notifications_mark_read_handler' );

/**
 * Helper for disable force friendship to messages the filter
 *
 * @since 1.0.0
 *
 * @return false
 */
function automatorwp_buddyboss_disable_force_friendship_to_message() {
    return false;
}

/**
 * Options callback for select2 fields assigned to reactions
 *
 * @since 1.0.0
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_buddyboss_options_cb_reaction( $field ) {

    // Setup vars
    $value = $field->escaped_value;
    $none_value = 'any';
    $none_label = __( 'any reaction', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );

    if( ! empty( $value ) ) {
        if( ! is_array( $value ) ) {
            $value = array( $value );
        }

        foreach( $value as $reaction_id ) {

            // Skip option none
            if( $reaction_id === $none_value ) {
                continue;
            }

            $options[$reaction_id] = automatorwp_buddyboss_get_reaction_title( $reaction_id );
        }
    }

    return $options;

}

/**
 * Get the reaction title
 *
 * @since 1.0.0
 *
 * @param int $reaction_id
 *
 * @return string|null
 */
function automatorwp_buddyboss_get_reaction_title( $reaction_id ) {

    // Empty title if no ID provided
    if( absint( $reaction_id ) === 0 ) {
        return '';
    }

    $reactions = array();

    // Get the reactions
    $reactions = bb_load_reaction()->bb_get_reactions( 'emotions' );

    foreach ( $reactions as $reaction ) {
        if ( absint( $reaction['id'] ) === absint( $reaction_id ) ) {
            return $reaction['icon_text']; 
        }
    }

    return $reactions;

}